#! /bin/sh
set -e

# Contextual checks
if [ -x "$PWD/scripts/test.sh" ] \
&& [ -r "$PWD/scripts/gen_test_carnet.pl" ] \
&& [ -x "$PWD/carnet" ] \
&& [ -x "$PWD/carnet_activity.pl" ]
then
	# We test from the root of the carnet project source directory.
	readonly CARNET_GENERATOR="$PWD/scripts/gen_test_carnet.pl"
	PATH="$PWD:$PATH"
elif [ -x "$PWD/test.sh" ] \
  && [ -r "$PWD/gen_test_carnet.pl" ] \
  && [ -x "$PWD/../carnet" ] \
  && [ -x "$PWD/../carnet_activity.pl" ]
then
	# We test from the scripts directory the carnet project source.
	readonly CARNET_GENERATOR="$PWD/gen_test_carnet.pl"
	PATH="$PWD/..:$PATH"
elif [ -r "$PWD/gen_test_carnet.pl" ]
then
	# We test from the scripts directory outside carnet source.
	readonly CARNET_GENERATOR="$PWD/gen_test_carnet.pl"
else
	cat >&2 <<-END
	error: failure to determine the context of the test.
	       Make sure to run test.sh from within the directory of
	       carnet source project.  If running the test independently
	       of the source directory, e.g. for system integration
	       testing, please make sure to ship the gen_test_carnet.pl
	       script along test.sh.
	END
	exit 1
fi

# Support functions

# capture_feuillet slurps its input and tries to return the first
# occurrence of a feuillet reference in its first line.  This is
# intended to capture a feuillet reference out of a carnet --search
# output.
capture_feuillet () {
	sed -n '1s/.*\([0-9-]\{10\}T[0-9:]\{5\}\).*/\1/p'
}

# Setting up test environment
readonly TEST_DIR="$(mktemp -d /tmp/carnet_dir.XXXXXX)"
trap 'rm -r "$TEST_DIR"' EXIT
CARNET_DIR="$TEST_DIR"
export CARNET_DIR
CARNET_EDITOR="ed"
export CARNET_EDITOR
CARNET_PAGER="cat"
export CARNET_PAGER
CARNET_WEEK_FILE="$CARNET_DIR/$(date +%G-%V).txt"

# Superficial checks
echo '$ carnet --version'
carnet --version
echo '$ carnet --help'
carnet --help

# Spawning a carnet out of the blue.
echo '$ carnet'
echo 'i
First entry
.
wq' \
| carnet

sed -i 's/^[0-9:]\{5\}/FF:FF/' "$CARNET_WEEK_FILE"

echo '$ carnet'
echo 'i
Second entry
.
wq' \
| carnet

sed -i 's/^[0-9:]\{5\}/EE:EE/' "$CARNET_WEEK_FILE"

echo '$ carnet'
echo 'i
Third entry
.
wq' \
| carnet

# Patching up timestamps to make them distinct.
sed -i 's/^[0-9:]\{5\}/DD:DD/' "$CARNET_WEEK_FILE"
sed -i 's/^FF:FF/00:01/' "$CARNET_WEEK_FILE"
sed -i 's/^EE:EE/12:33/' "$CARNET_WEEK_FILE"
sed -i 's/^DD:DD/22:34/' "$CARNET_WEEK_FILE"

echo '$ carnet  # Adding a todo item 1'
echo 'i
TODO: item todo number one.
.
wq' \
| carnet

echo '$ carnet  # Adding resolved item 1'
echo 'i
[*] This issue is resolved now.  :)
.
wq' \
| carnet

echo '$ carnet  # Adding a todo item 2'
echo 'i
[ ] item todo number two.
.
wq' \
| carnet

echo '$ carnet  # Adding resolved item 2'
echo 'i
DONE: at last.
.
wq' \
| carnet

echo '$ carnet  # Adding a pending todo item'
echo 'i
[pending] item pending resolution of something else.
.
wq' \
| carnet

# Checking the search engine
echo '$ carnet --search nothing'
carnet --search nothing

echo '$ carnet -s First'
carnet -s First

echo '$ carnet -s Second'
carnet -s Second

echo '$ carnet -s Third'
carnet -s Third

echo '$ carnet -s entry'
carnet -s entry

echo '$ carnet --todo'
carnet --todo

echo '$ carnet --todo pending'
carnet --todo pending

echo '$ carnet --done'
carnet --done

echo '$ carnet --readonly'
carnet --readonly

# Checking general feuillets handling
readonly FEUILLET1="$( \
	carnet -s First \
	| capture_feuillet
)"

readonly FEUILLET2="$( \
	carnet -s Second \
	| capture_feuillet
)"

readonly FEUILLET3="$( \
	carnet -s Third \
	| capture_feuillet
)"

echo '$ carnet --feuillet '"$FEUILLET1"
carnet --feuillet "$FEUILLET1"

echo '$ carnet --feuillet '"$FEUILLET2"
carnet --feuillet "$FEUILLET2"

echo '$ carnet --feuillet '"$FEUILLET3"
carnet --feuillet "$FEUILLET3"

echo '$ carnet -ef '"$FEUILLET1"
echo 'a
This is a line injected at the location of the editor pointer.
.
wq' \
| carnet -ef "$FEUILLET1"

echo '$ carnet --feuillet '"$FEUILLET1"
carnet --feuillet "$FEUILLET1"

# Activity module check
echo '$ carnet -a'
carnet -a
set +e
echo '$ CARNET_DIR="/nonexistent" carnet -a'
CARNET_DIR="/nonexistent" carnet -a
if [ "$?" != 1 ]
then
	cat >&2 <<END
error: the following command should have been failing:
       $ CARNET_DIR="/nonexistent" carnet -a
END
	exit 1
fi
set -e

# Module support checks
carnet -lm
carnet --list-modules
carnet --list-module
echo -n '#!/bin/bash\necho "Running module1"\n' > "$CARNET_DIR"/module1
echo -n '#!/bin/bash\necho "Running module2"\n' > "$CARNET_DIR"/module2
echo -n '#!/bin/bash\necho "Running module3"\n' > "$CARNET_DIR"/module3
chmod +x "$CARNET_DIR"/module1
chmod +x "$CARNET_DIR"/module2
chmod +x "$CARNET_DIR"/module3
carnet -lm
carnet --list-modules
carnet --list-module
carnet -m module1
carnet -m module2
carnet -m module3

echo "$ # Padding the carnet with very many entries..."
readonly ORIGDIR="$PWD"
cd "$CARNET_DIR"
perl "$CARNET_GENERATOR" 2024 2025
cd "$ORIGDIR"

echo "$ carnet --activity"
carnet --activity

# Populating the register.txt: this is done last because it induces a
# hard delay of 5 seconds.  If something has to fail, it would be nice
# that it fails before the delayed command.
echo '$ carnet --edit-feuillet'
echo "i
Register of feuillets of interest
=================================

$FEUILLET1: first feuillet
$FEUILLET2: second feuillet
$FEUILLET3: third feuillet
.
wq" \
| carnet --edit-feuillet

echo '$ carnet --feuillet'
carnet --feuillet

echo '$ # All tests passed OK.'
